<?php
/**
 * WHMCS CashApp Payment Gateway Callback Handler
 *
 * This file handles webhook callbacks from CashApp when payment status changes
 */

require_once __DIR__ . '/../../../init.php';
require_once __DIR__ . '/../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../includes/invoicefunctions.php';
require_once __DIR__ . '/../cashapp/cashapp.php';

use WHMCS\Database\Capsule;

// Get gateway configuration
$gatewayModuleName = basename(__FILE__, '.php');
$gatewayParams = getGatewayVariables($gatewayModuleName);

if (!$gatewayParams['type']) {
    die("Module Not Activated");
}

// Get webhook payload
$rawPayload = file_get_contents('php://input');
$webhookData = json_decode($rawPayload, true);

// Verify webhook signature
$signature = $_SERVER['HTTP_X_CASHAPP_SIGNATURE'] ?? '';

if (!CashAppWebhook::verifySignature($rawPayload, $signature, $gatewayParams['webhookSecret'])) {
    logTransaction($gatewayModuleName, $_REQUEST, "Invalid Webhook Signature");
    http_response_code(401);
    die("Invalid signature");
}

// Extract webhook data
$eventType = $webhookData['event_type'] ?? '';
$cashappInvoiceId = $webhookData['invoice_id'] ?? '';
$paymentId = $webhookData['payment_id'] ?? '';
$status = $webhookData['status'] ?? '';
$amount = $webhookData['amount'] ?? 0;
$currency = $webhookData['currency'] ?? '';
$metadata = $webhookData['metadata'] ?? array();

// Log the webhook
if ($gatewayParams['debugMode']) {
    logTransaction($gatewayModuleName, $webhookData, "Webhook Received: " . $eventType);
}

// Get WHMCS invoice ID from database
$invoiceRecord = CashAppDatabase::getInvoiceByCashAppId($cashappInvoiceId);

if (!$invoiceRecord) {
    logTransaction($gatewayModuleName, $webhookData, "Invoice not found in database");
    http_response_code(404);
    die("Invoice not found");
}

$invoiceId = $invoiceRecord->whmcs_invoice_id;

// Check if invoice exists in WHMCS
$invoiceData = Capsule::table('tblinvoices')
    ->where('id', $invoiceId)
    ->first();

if (!$invoiceData) {
    logTransaction($gatewayModuleName, $webhookData, "WHMCS Invoice #{$invoiceId} not found");
    http_response_code(404);
    die("WHMCS Invoice not found");
}

// Process based on event type
switch ($eventType) {
    case 'payment.completed':
        handlePaymentCompleted($invoiceId, $paymentId, $amount, $currency, $cashappInvoiceId, $gatewayModuleName, $gatewayParams);
        break;

    case 'payment.failed':
        handlePaymentFailed($invoiceId, $cashappInvoiceId, $webhookData, $gatewayModuleName, $gatewayParams);
        break;

    case 'invoice.expired':
        handleInvoiceExpired($invoiceId, $cashappInvoiceId, $gatewayModuleName, $gatewayParams);
        break;

    default:
        logTransaction($gatewayModuleName, $webhookData, "Unknown event type: " . $eventType);
}

http_response_code(200);
echo json_encode(array('success' => true));
exit;

/**
 * Handle successful payment
 */
function handlePaymentCompleted($invoiceId, $paymentId, $amount, $currency, $cashappInvoiceId, $gatewayName, $gatewayParams)
{
    // Check if payment already processed
    $existingTransaction = Capsule::table('tblaccounts')
        ->where('invoiceid', $invoiceId)
        ->where('transid', $paymentId)
        ->first();

    if ($existingTransaction) {
        logTransaction($gatewayName, array('invoice_id' => $invoiceId, 'payment_id' => $paymentId), "Payment already processed");
        return;
    }

    // Get invoice details
    $invoice = Capsule::table('tblinvoices')
        ->where('id', $invoiceId)
        ->first();

    if (!$invoice) {
        return;
    }

    // Verify amount matches
    $invoiceTotal = $invoice->total;
    if (abs($invoiceTotal - $amount) > 0.01) {
        logTransaction($gatewayName, array(
            'invoice_id' => $invoiceId,
            'expected' => $invoiceTotal,
            'received' => $amount
        ), "Amount mismatch");
        return;
    }

    // Add payment to invoice
    $addInvoicePayment = array(
        'invoiceid' => $invoiceId,
        'transid' => $paymentId,
        'gateway' => $gatewayName,
        'date' => date('Y-m-d H:i:s'),
        'amount' => $amount,
        'fees' => 0,
    );

    $result = localAPI('AddInvoicePayment', $addInvoicePayment);

    // Update invoice record in database
    CashAppDatabase::updateInvoiceStatus($cashappInvoiceId, 'paid', $paymentId);

    // Log success
    logTransaction($gatewayName, array(
        'invoice_id' => $invoiceId,
        'payment_id' => $paymentId,
        'amount' => $amount,
        'api_result' => $result
    ), "Payment Successful");

    // Send payment confirmation email
    if ($invoice->userid) {
        sendMessage('Invoice Payment Confirmation', $invoice->userid, array('invoiceid' => $invoiceId));
    }
}

/**
 * Handle failed payment
 */
function handlePaymentFailed($invoiceId, $cashappInvoiceId, $webhookData, $gatewayName, $gatewayParams)
{
    // Update invoice record
    CashAppDatabase::updateInvoiceStatus($cashappInvoiceId, 'failed');

    // Log failure
    logTransaction($gatewayName, $webhookData, "Payment Failed - Invoice #{$invoiceId}");

    // Optional: Send notification to admin
    if ($gatewayParams['debugMode']) {
        $failureReason = $webhookData['failure_reason'] ?? 'Unknown';
        logActivity("CashApp payment failed for Invoice #{$invoiceId}. Reason: {$failureReason}");
    }
}

/**
 * Handle expired invoice
 */
function handleInvoiceExpired($invoiceId, $cashappInvoiceId, $gatewayName, $gatewayParams)
{
    // Update invoice record
    CashAppDatabase::updateInvoiceStatus($cashappInvoiceId, 'expired');

    // Log expiration
    logTransaction($gatewayName, array('invoice_id' => $invoiceId, 'cashapp_invoice_id' => $cashappInvoiceId), "Invoice Expired");

    // Optional: Send reminder to customer
    $invoice = Capsule::table('tblinvoices')
        ->where('id', $invoiceId)
        ->first();

    if ($invoice && $invoice->userid && $invoice->status === 'Unpaid') {
        // You can send a custom email template here
        // sendMessage('Invoice Payment Reminder', $invoice->userid, array('invoiceid' => $invoiceId));
    }
}
