<?php
/**
 * WHMCS CashApp Payment Gateway Module
 *
 * @copyright Copyright (c) 2024
 * @license https://www.whmcs.com/license/ WHMCS Eula
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

// Include helper classes
require_once __DIR__ . '/cashapp/cashapp.php';

/**
 * Define module related meta data.
 *
 * @return array
 */
function cashapp_MetaData()
{
    return array(
        'DisplayName' => 'CashApp',
        'APIVersion' => '1.1',
        'DisableLocalCredtCardInput' => true,
        'TokenisedStorage' => false,
    );
}

/**
 * Define gateway configuration options.
 *
 * @return array
 */
function cashapp_config()
{
    return array(
        'FriendlyName' => array(
            'Type' => 'System',
            'Value' => 'CashApp',
        ),
        'apiKey' => array(
            'FriendlyName' => 'API Key',
            'Type' => 'text',
            'Size' => '50',
            'Default' => '',
            'Description' => 'Enter your CashApp API Key',
        ),
        'apiSecret' => array(
            'FriendlyName' => 'API Secret',
            'Type' => 'password',
            'Size' => '50',
            'Default' => '',
            'Description' => 'Enter your CashApp API Secret',
        ),
        'webhookSecret' => array(
            'FriendlyName' => 'Webhook Secret',
            'Type' => 'password',
            'Size' => '50',
            'Default' => '',
            'Description' => 'Enter your CashApp Webhook Secret for signature verification',
        ),
        'cashtag' => array(
            'FriendlyName' => 'CashTag',
            'Type' => 'text',
            'Size' => '30',
            'Default' => '',
            'Description' => 'Your CashApp $Cashtag (without the $ symbol)',
        ),
        'environment' => array(
            'FriendlyName' => 'Environment',
            'Type' => 'dropdown',
            'Options' => array(
                'sandbox' => 'Sandbox (Testing)',
                'production' => 'Production (Live)',
            ),
            'Default' => 'sandbox',
            'Description' => 'Select the environment mode',
        ),
        'invoiceExpiry' => array(
            'FriendlyName' => 'Invoice Expiry Time',
            'Type' => 'text',
            'Size' => '10',
            'Default' => '3600',
            'Description' => 'Invoice expiration time in seconds (default: 3600 = 1 hour)',
        ),
        'showQRCode' => array(
            'FriendlyName' => 'Show QR Code',
            'Type' => 'yesno',
            'Description' => 'Display QR code on payment page',
            'Default' => 'yes',
        ),
        'debugMode' => array(
            'FriendlyName' => 'Debug Mode',
            'Type' => 'yesno',
            'Description' => 'Enable debug logging',
            'Default' => 'no',
        ),
    );
}

/**
 * Payment link.
 *
 * @param array $params Payment Gateway Module Parameters
 * @return string
 */
function cashapp_link($params)
{
    // Gateway Configuration Parameters
    $apiKey = $params['apiKey'];
    $apiSecret = $params['apiSecret'];
    $cashtag = $params['cashtag'];
    $environment = $params['environment'];
    $showQRCode = $params['showQRCode'];
    $invoiceExpiry = $params['invoiceExpiry'];
    $debugMode = $params['debugMode'];

    // Invoice Parameters
    $invoiceId = $params['invoiceid'];
    $description = $params["description"];
    $amount = $params['amount'];
    $currency = $params['currency'];

    // Client Parameters
    $firstname = $params['clientdetails']['firstname'];
    $lastname = $params['clientdetails']['lastname'];
    $email = $params['clientdetails']['email'];
    $address1 = $params['clientdetails']['address1'];
    $city = $params['clientdetails']['city'];
    $state = $params['clientdetails']['state'];
    $postcode = $params['clientdetails']['postcode'];
    $country = $params['clientdetails']['country'];
    $phone = $params['clientdetails']['phonenumber'];

    // System Parameters
    $companyName = $params['companyname'];
    $systemUrl = $params['systemurl'];
    $returnUrl = $params['returnurl'];
    $langPayNow = $params['langpaynow'];
    $moduleDisplayName = $params['name'];
    $moduleName = $params['paymentmethod'];

    // Callback URL
    $callbackUrl = $systemUrl . 'modules/gateways/callback/' . $moduleName . '.php';

    // Create CashApp invoice
    try {
        $invoice = cashapp_createInvoice(
            $apiKey,
            $apiSecret,
            $environment,
            $invoiceId,
            $amount,
            $currency,
            $description,
            $email,
            $callbackUrl,
            $invoiceExpiry,
            $debugMode
        );

        if (!$invoice || !isset($invoice['payment_url'])) {
            throw new Exception('Failed to create CashApp invoice');
        }

        // Store invoice data in database for tracking
        cashapp_storeInvoiceData($invoiceId, $invoice);

        $paymentUrl = $invoice['payment_url'];
        $qrCodeUrl = isset($invoice['qr_code_url']) ? $invoice['qr_code_url'] : '';
        $cashappInvoiceId = $invoice['id'];

        // Build payment form HTML
        $htmlOutput = '<div class="cashapp-payment-container" style="text-align: center; padding: 20px;">';
        $htmlOutput .= '<h3>Pay with CashApp</h3>';
        $htmlOutput .= '<p>Invoice #' . $invoiceId . '</p>';
        $htmlOutput .= '<p><strong>Amount: ' . $currency . ' ' . number_format($amount, 2) . '</strong></p>';
        
        if ($showQRCode && $qrCodeUrl) {
            $htmlOutput .= '<div style="margin: 20px 0;">';
            $htmlOutput .= '<p>Scan this QR code with your CashApp:</p>';
            $htmlOutput .= '<img src="' . htmlspecialchars($qrCodeUrl) . '" alt="CashApp QR Code" style="max-width: 300px; border: 2px solid #00D632; padding: 10px; border-radius: 10px;">';
            $htmlOutput .= '</div>';
            $htmlOutput .= '<p style="margin: 20px 0;"><strong>OR</strong></p>';
        }
        
        $htmlOutput .= '<a href="' . htmlspecialchars($paymentUrl) . '" class="btn btn-success btn-lg" style="background-color: #00D632; border: none; padding: 15px 40px; font-size: 18px; text-decoration: none; color: white; border-radius: 5px; display: inline-block;">';
        $htmlOutput .= '<i class="fas fa-dollar-sign"></i> Pay Now with CashApp';
        $htmlOutput .= '</a>';
        
        $htmlOutput .= '<p style="margin-top: 20px; font-size: 12px; color: #666;">You will be redirected to CashApp to complete your payment</p>';
        $htmlOutput .= '<p style="font-size: 12px; color: #666;">Invoice expires in ' . ($invoiceExpiry / 60) . ' minutes</p>';
        $htmlOutput .= '</div>';

        return $htmlOutput;

    } catch (Exception $e) {
        if ($debugMode) {
            logTransaction($moduleName, $_REQUEST, "Error: " . $e->getMessage());
        }
        return '<div class="alert alert-danger">Unable to generate CashApp payment link. Please contact support.</div>';
    }
}

/**
 * Create CashApp invoice via API
 */
function cashapp_createInvoice($apiKey, $apiSecret, $environment, $invoiceId, $amount, $currency, $description, $customerEmail, $callbackUrl, $expiresIn, $debug)
{
    try {
        $api = new CashAppAPI($apiKey, $apiSecret, $environment, $debug);
        
        $invoiceData = array(
            'amount' => floatval($amount),
            'currency' => $currency,
            'description' => $description,
            'customer_email' => $customerEmail,
            'callback_url' => $callbackUrl,
            'expires_in' => intval($expiresIn),
            'metadata' => array(
                'whmcs_invoice_id' => $invoiceId,
                'source' => 'whmcs'
            )
        );

        return $api->createInvoice($invoiceData);
        
    } catch (Exception $e) {
        if ($debug) {
            logTransaction('cashapp', array('error' => $e->getMessage()), "Create Invoice Error");
        }
        return false;
    }
}

/**
 * Store invoice data in custom table
 */
function cashapp_storeInvoiceData($whmcsInvoiceId, $invoiceData)
{
    return CashAppDatabase::storeInvoice($whmcsInvoiceId, $invoiceData);
}

/**
 * Refund transaction.
 *
 * @param array $params Payment Gateway Module Parameters
 * @return array Transaction response status
 */
function cashapp_refund($params)
{
    $apiKey = $params['apiKey'];
    $apiSecret = $params['apiSecret'];
    $environment = $params['environment'];
    $debugMode = $params['debugMode'];

    $transactionId = $params['transid'];
    $refundAmount = $params['amount'];
    $currency = $params['currency'];

    try {
        $api = new CashAppAPI($apiKey, $apiSecret, $environment, $debugMode);
        
        $refundData = array(
            'payment_id' => $transactionId,
            'amount' => floatval($refundAmount),
            'currency' => $currency,
            'reason' => 'Refund requested via WHMCS'
        );

        $result = $api->createRefund($refundData);

        return array(
            'status' => 'success',
            'rawdata' => $result,
            'transid' => $result['refund_id'],
            'fees' => 0,
        );

    } catch (Exception $e) {
        if ($debugMode) {
            logTransaction('cashapp', array('error' => $e->getMessage()), "Refund Error");
        }

        return array(
            'status' => 'declined',
            'rawdata' => array('error' => $e->getMessage()),
        );
    }
}
