<?php
/**
 * WHMCS CashApp Payment Gateway - Configuration File
 * 
 * This file contains helper functions and configuration
 * Location: /modules/gateways/cashapp/cashapp.php
 *
 * @copyright Copyright (c) 2024
 * @license https://www.whmcs.com/license/ WHMCS Eula
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

/**
 * CashApp API Client Class
 */
class CashAppAPI
{
    private $apiKey;
    private $apiSecret;
    private $environment;
    private $baseUrl;
    private $debug;

    public function __construct($apiKey, $apiSecret, $environment = 'sandbox', $debug = false)
    {
        $this->apiKey = $apiKey;
        $this->apiSecret = $apiSecret;
        $this->environment = $environment;
        $this->debug = $debug;
        
        $this->baseUrl = ($environment === 'production') 
            ? 'https://api.cash.app/v1' 
            : 'https://sandbox.api.cash.app/v1';
    }

    /**
     * Generate authentication headers
     */
    private function getAuthHeaders()
    {
        $timestamp = time();
        $signature = hash_hmac('sha256', $timestamp . $this->apiKey, $this->apiSecret);

        return array(
            'Authorization: Bearer ' . $this->apiKey,
            'X-CashApp-Timestamp: ' . $timestamp,
            'X-CashApp-Signature: ' . $signature,
            'Content-Type: application/json',
            'Accept: application/json'
        );
    }

    /**
     * Make API request
     */
    private function request($method, $endpoint, $data = null)
    {
        $url = $this->baseUrl . $endpoint;
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $this->getAuthHeaders());
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'PUT') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($this->debug) {
            logTransaction('cashapp', array(
                'method' => $method,
                'endpoint' => $endpoint,
                'request' => $data,
                'response' => $response,
                'http_code' => $httpCode,
                'error' => $error
            ), "CashApp API Request");
        }

        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }

        $result = json_decode($response, true);
        
        if ($httpCode >= 400) {
            $errorMessage = isset($result['error']['message']) 
                ? $result['error']['message'] 
                : "API Error (HTTP {$httpCode})";
            throw new Exception($errorMessage);
        }

        return $result;
    }

    /**
     * Create invoice
     */
    public function createInvoice($data)
    {
        return $this->request('POST', '/invoices', $data);
    }

    /**
     * Get invoice details
     */
    public function getInvoice($invoiceId)
    {
        return $this->request('GET', '/invoices/' . $invoiceId);
    }

    /**
     * Cancel invoice
     */
    public function cancelInvoice($invoiceId)
    {
        return $this->request('POST', '/invoices/' . $invoiceId . '/cancel');
    }

    /**
     * Create refund
     */
    public function createRefund($data)
    {
        return $this->request('POST', '/refunds', $data);
    }

    /**
     * Get refund details
     */
    public function getRefund($refundId)
    {
        return $this->request('GET', '/refunds/' . $refundId);
    }

    /**
     * Get payment details
     */
    public function getPayment($paymentId)
    {
        return $this->request('GET', '/payments/' . $paymentId);
    }

    /**
     * List invoices
     */
    public function listInvoices($params = array())
    {
        $queryString = http_build_query($params);
        $endpoint = '/invoices' . ($queryString ? '?' . $queryString : '');
        return $this->request('GET', $endpoint);
    }
}

/**
 * Database Helper Functions
 */
class CashAppDatabase
{
    /**
     * Ensure custom table exists
     */
    public static function ensureTableExists()
    {
        $tableName = 'mod_cashapp_invoices';
        
        $tableExists = Capsule::schema()->hasTable($tableName);
        
        if (!$tableExists) {
            Capsule::schema()->create($tableName, function ($table) {
                $table->increments('id');
                $table->integer('whmcs_invoice_id')->index();
                $table->string('cashapp_invoice_id', 255)->unique();
                $table->decimal('amount', 10, 2);
                $table->string('currency', 3);
                $table->string('status', 50)->default('pending')->index();
                $table->text('payment_url')->nullable();
                $table->text('qr_code_url')->nullable();
                $table->string('payment_id', 255)->nullable();
                $table->timestamp('paid_at')->nullable();
                $table->timestamps();
            });
            
            logActivity("CashApp: Created database table {$tableName}");
        }
    }

    /**
     * Store invoice data
     */
    public static function storeInvoice($whmcsInvoiceId, $cashappData)
    {
        self::ensureTableExists();
        
        return Capsule::table('mod_cashapp_invoices')->insert([
            'whmcs_invoice_id' => $whmcsInvoiceId,
            'cashapp_invoice_id' => $cashappData['id'],
            'amount' => $cashappData['amount'],
            'currency' => $cashappData['currency'],
            'status' => 'pending',
            'payment_url' => $cashappData['payment_url'] ?? null,
            'qr_code_url' => $cashappData['qr_code_url'] ?? null,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Get invoice by CashApp invoice ID
     */
    public static function getInvoiceByCashAppId($cashappInvoiceId)
    {
        return Capsule::table('mod_cashapp_invoices')
            ->where('cashapp_invoice_id', $cashappInvoiceId)
            ->first();
    }

    /**
     * Get invoice by WHMCS invoice ID
     */
    public static function getInvoiceByWHMCSId($whmcsInvoiceId)
    {
        return Capsule::table('mod_cashapp_invoices')
            ->where('whmcs_invoice_id', $whmcsInvoiceId)
            ->orderBy('created_at', 'desc')
            ->first();
    }

    /**
     * Update invoice status
     */
    public static function updateInvoiceStatus($cashappInvoiceId, $status, $paymentId = null)
    {
        $data = [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        if ($paymentId) {
            $data['payment_id'] = $paymentId;
        }

        if ($status === 'paid') {
            $data['paid_at'] = date('Y-m-d H:i:s');
        }

        return Capsule::table('mod_cashapp_invoices')
            ->where('cashapp_invoice_id', $cashappInvoiceId)
            ->update($data);
    }

    /**
     * Get invoice statistics
     */
    public static function getStatistics($days = 30)
    {
        $since = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $stats = Capsule::table('mod_cashapp_invoices')
            ->where('created_at', '>=', $since)
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(CASE WHEN status = "paid" THEN 1 ELSE 0 END) as paid_invoices,
                SUM(CASE WHEN status = "pending" THEN 1 ELSE 0 END) as pending_invoices,
                SUM(CASE WHEN status = "failed" THEN 1 ELSE 0 END) as failed_invoices,
                SUM(CASE WHEN status = "expired" THEN 1 ELSE 0 END) as expired_invoices,
                SUM(CASE WHEN status = "paid" THEN amount ELSE 0 END) as total_amount_paid
            ')
            ->first();

        return $stats;
    }

    /**
     * Clean up old records
     */
    public static function cleanupOldRecords($days = 90)
    {
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $deleted = Capsule::table('mod_cashapp_invoices')
            ->where('created_at', '<', $cutoffDate)
            ->whereIn('status', ['expired', 'cancelled', 'failed'])
            ->delete();

        if ($deleted > 0) {
            logActivity("CashApp: Cleaned up {$deleted} old invoice records");
        }

        return $deleted;
    }
}

/**
 * Webhook Verification Helper
 */
class CashAppWebhook
{
    /**
     * Verify webhook signature
     */
    public static function verifySignature($payload, $signature, $secret)
    {
        if (empty($signature) || empty($secret)) {
            return false;
        }

        $expectedSignature = hash_hmac('sha256', $payload, $secret);
        
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Parse webhook payload
     */
    public static function parsePayload($rawPayload)
    {
        $data = json_decode($rawPayload, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON payload: ' . json_last_error_msg());
        }

        return $data;
    }

    /**
     * Validate webhook data
     */
    public static function validateWebhookData($data)
    {
        $required = ['event_type', 'invoice_id'];
        
        foreach ($required as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                throw new Exception("Missing required field: {$field}");
            }
        }

        return true;
    }
}

/**
 * Utility Functions
 */
class CashAppUtils
{
    /**
     * Format amount for display
     */
    public static function formatAmount($amount, $currency = 'USD')
    {
        return $currency . ' ' . number_format($amount, 2);
    }

    /**
     * Generate unique reference ID
     */
    public static function generateReferenceId($invoiceId)
    {
        return 'WHMCS-' . $invoiceId . '-' . time();
    }

    /**
     * Validate amount
     */
    public static function validateAmount($amount)
    {
        return is_numeric($amount) && $amount > 0;
    }

    /**
     * Get callback URL
     */
    public static function getCallbackUrl($systemUrl, $moduleName = 'cashapp')
    {
        return rtrim($systemUrl, '/') . '/modules/gateways/callback/' . $moduleName . '.php';
    }

    /**
     * Log transaction with additional context
     */
    public static function logTransaction($gateway, $data, $message, $level = 'info')
    {
        $logData = array_merge([
            'timestamp' => date('Y-m-d H:i:s'),
            'level' => $level,
            'message' => $message,
        ], $data);

        logTransaction($gateway, $logData, $message);
    }

    /**
     * Send admin notification
     */
    public static function sendAdminNotification($subject, $message)
    {
        $adminEmail = Capsule::table('tbladmins')
            ->where('roleid', 1)
            ->value('email');

        if ($adminEmail) {
            $command = 'SendEmail';
            $postData = [
                'messagename' => 'Admin Notification',
                'customsubject' => '[CashApp Gateway] ' . $subject,
                'custommessage' => $message,
                'customtype' => 'general',
                'customto' => $adminEmail,
            ];

            try {
                localAPI($command, $postData);
            } catch (Exception $e) {
                logActivity('CashApp: Failed to send admin notification - ' . $e->getMessage());
            }
        }
    }
}

/**
 * Admin Widget Functions
 */
class CashAppAdmin
{
    /**
     * Get dashboard statistics
     */
    public static function getDashboardStats()
    {
        $stats30Days = CashAppDatabase::getStatistics(30);
        $stats7Days = CashAppDatabase::getStatistics(7);
        $statsToday = CashAppDatabase::getStatistics(1);

        return [
            'today' => $statsToday,
            'last_7_days' => $stats7Days,
            'last_30_days' => $stats30Days,
        ];
    }

    /**
     * Get pending invoices
     */
    public static function getPendingInvoices()
    {
        return Capsule::table('mod_cashapp_invoices')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
    }

    /**
     * Get recent payments
     */
    public static function getRecentPayments($limit = 10)
    {
        return Capsule::table('mod_cashapp_invoices')
            ->where('status', 'paid')
            ->orderBy('paid_at', 'desc')
            ->limit($limit)
            ->get();
    }
}
