<?php
/**
 * 1Stream B2B WHMCS Provisioning Module
 * For managing reseller credits and service packages
 *
 * @copyright Copyright (c) 2024
 * @license MIT
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

/**
 * Define module metadata
 */
function onestream_MetaData()
{
    return array(
        'DisplayName' => '1Stream B2B',
        'APIVersion' => '1.1',
        'RequiresServer' => true,
        'DefaultNonSSLPort' => '80',
        'DefaultSSLPort' => '443',
        'ServiceSingleSignOnLabel' => 'Login to 1Stream Panel',
    );
}

/**
 * Define product configuration options
 */
function onestream_ConfigOptions($params)
{
    // Hardcoded KromeTV packages
    $packageOptions = array(
        '' => '-- Select Package --',
        119 => 'ID 119: trial (0 credits, 1 day, 2 conn)',
        126 => 'ID 126: 1 month kids only (0.5 credits, 30 days, 2 conn)',
        128 => 'ID 128: 1 month 2 connections Live w/Adult (1 credits, 30 days, 2 conn)',
        101 => 'ID 101: 1 Month 2 connections w/Adult (1 credits, 30 days, 2 conn)',
        137 => 'ID 137: 1 Month MAG w/Adult (1 credits, 30 days, 2 conn)',
        132 => 'ID 132: 1 Month 3 connections w/Adult (1.5 credits, 30 days, 3 conn)',
        103 => 'ID 103: 1 Month 4 connections w/Adult (2 credits, 30 days, 4 conn)',
        105 => 'ID 105: 1 Month 6 connections w/Adult (3 credits, 30 days, 6 conn)',
        134 => 'ID 134: 1 Month 8 connections w/Adult (4.5 credits, 30 days, 8 conn)',
        107 => 'ID 107: 3 Month 2 connections w/Adult (3 credits, 90 days, 2 conn)',
        109 => 'ID 109: 3 Month 4 connections w/Adult (6 credits, 90 days, 4 conn)',
        111 => 'ID 111: 3 Month 6 connections w/Adult (8 credits, 90 days, 6 conn)',
        120 => 'ID 120: 6 month 2 connections w/Adult (6 credits, 180 days, 2 conn)',
        122 => 'ID 122: 6 month 4 connections w/Adult (12 credits, 180 days, 4 conn)',
        124 => 'ID 124: 6 month 6 connections w/Adult (16 credits, 180 days, 6 conn)',
        113 => 'ID 113: 12 Month 2 connections w/Adult (12 credits, 365 days, 2 conn)',
        115 => 'ID 115: 12 Month 4 connections w/Adult (20 credits, 365 days, 4 conn)',
        129 => 'ID 129: 12 month 2 connections Live w/Adult (12 credits, 365 days, 2 conn)',
        117 => 'ID 117: 12 Month 6 connections w/Adult (30 credits, 365 days, 6 conn)',
        139 => 'ID 139: 1 month 2 connections 24/7 + VOD (1 credits, 30 days, 2 conn)',
        136 => 'ID 136: HDO/Stremio 1 Month (0 credits, 30 days, 1 conn)',
    );
    
    // Always log when this function is called
    logModuleCall('onestream', 'ConfigOptions Called', array(
        'serverid' => $params['serverid'] ?? 'not set',
        'params_keys' => array_keys($params),
    ), null, null);
    
    if (count($packageOptions) == 1) {
        $packageOptions['manual'] = '-- Enter Package ID manually below --';
    }
    
    return array(
        'Service Type' => array(
            'Type' => 'dropdown',
            'Options' => array(
                'line' => 'Line/Subscription',
                'credits' => 'Reseller Credits',
            ),
            'Description' => 'Type of service to provision',
            'Default' => 'line',
        ),
        'Package' => array(
            'Type' => 'dropdown',
            'Options' => $packageOptions,
            'Description' => '⚠️ FOR LINE/SUBSCRIPTION ONLY - Select package',
        ),
        'Credit Amount' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '0',
            'Description' => '⚠️ FOR RESELLER CREDITS ONLY - Enter credit amount (leave 0 for Line/Subscription)',
        ),
        'Is Trial' => array(
            'Type' => 'yesno',
            'Description' => '⚠️ FOR LINE/SUBSCRIPTION ONLY - Mark as trial',
        ),
    );
}

/**
 * Provision a new account/service
 */
function onestream_CreateAccount(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            $result = onestream_AddCreditsToReseller($params);
        } else {
            $result = onestream_CreateLine($params);
        }
        
        if ($result['success']) {
            if (isset($result['line_id'])) {
                Capsule::table('tblhosting')
                    ->where('id', $params['serviceid'])
                    ->update([
                        'username' => $result['username'],
                        'password' => encrypt($result['password']),
                    ]);
            }
            return 'success';
        }
        
        return $result['error'] ?? 'Unknown error during provisioning';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'CreateAccount', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Create a new line
 */
function onestream_CreateLine($params)
{
    // Get package ID from dropdown (configoption2)
    $packageId = $params['configoption2'];
    
    if (empty($packageId) || $packageId == 'manual') {
        return array(
            'success' => false,
            'error' => 'Please select a package from the dropdown',
        );
    }
    
    $isTrial = $params['configoption4'] == 'on';
    
    $postData = array(
        'package' => intval($packageId),
        'reseller_notes' => 'WHMCS Service #' . $params['serviceid'],
        'rid' => 'whmcs_' . $params['serviceid'] . '_' . time(),
    );
    
    if ($isTrial) {
        $postData['is_trial'] = true;
    }
    
    $response = onestream_ApiCall($params, 'POST', '/ext/line/create', $postData);
    
    if (!isset($response['line_id'])) {
        return array(
            'success' => false,
            'error' => $response['error'] ?? 'Failed to create line',
        );
    }
    
    $lineId = $response['line_id'];
    
    try {
        sleep(1);
        
        $lines = onestream_ApiCall($params, 'GET', '/ext/lines');
        
        if (is_array($lines)) {
            foreach ($lines as $line) {
                if (isset($line['line_id']) && $line['line_id'] === $lineId) {
                    return array(
                        'success' => true,
                        'line_id' => $line['line_id'],
                        'username' => $line['username'],
                        'password' => $line['password'],
                        'expire_at' => $line['expire_at'] ?? null,
                    );
                }
            }
        }
        
        return array(
            'success' => false,
            'error' => 'Line created (ID: ' . $lineId . ') but could not retrieve credentials. Check 1Stream panel and update manually.',
        );
        
    } catch (Exception $e) {
        return array(
            'success' => false,
            'error' => 'Line created but error: ' . $e->getMessage(),
        );
    }
}

/**
 * Add credits to reseller
 */
function onestream_AddCreditsToReseller($params)
{
    $creditAmount = intval($params['configoption3']);
    
    if ($creditAmount <= 0) {
        return array('success' => false, 'error' => 'Invalid credit amount');
    }
    
    $profile = onestream_ApiCall($params, 'GET', '/ext/profile');
    
    if (isset($profile['credits'])) {
        return array(
            'success' => true,
            'credits' => $profile['credits'],
        );
    }
    
    return array(
        'success' => false,
        'error' => $profile['error'] ?? 'Failed to add credits',
    );
}

/**
 * Suspend a service
 */
function onestream_SuspendAccount(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            return 'success';
        }
        
        if (empty($params['username'])) {
            return 'Cannot suspend - no username found';
        }
        
        $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
            'username' => $params['username'],
            'password' => $params['password'],
        ));
        
        if (!isset($findResponse['line_id'])) {
            return 'Line not found';
        }
        
        $lineId = $findResponse['line_id'];
        
        $response = onestream_ApiCall($params, 'POST', "/ext/line/{$lineId}/disable");
        
        if (isset($response['line_id'])) {
            return 'success';
        }
        
        return $response['error'] ?? 'Failed to suspend line';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'SuspendAccount', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Unsuspend a service
 */
function onestream_UnsuspendAccount(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            return 'success';
        }
        
        if (empty($params['username'])) {
            return 'Cannot unsuspend - no username found';
        }
        
        $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
            'username' => $params['username'],
            'password' => $params['password'],
        ));
        
        if (!isset($findResponse['line_id'])) {
            return 'Line not found';
        }
        
        $lineId = $findResponse['line_id'];
        
        $response = onestream_ApiCall($params, 'POST', "/ext/line/{$lineId}/enable");
        
        if (isset($response['line_id'])) {
            return 'success';
        }
        
        return $response['error'] ?? 'Failed to unsuspend line';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'UnsuspendAccount', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Terminate a service
 */
function onestream_TerminateAccount(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            return 'success';
        }
        
        if (empty($params['username'])) {
            return 'success';
        }
        
        $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
            'username' => $params['username'],
            'password' => $params['password'],
        ));
        
        if (!isset($findResponse['line_id'])) {
            return 'success';
        }
        
        $lineId = $findResponse['line_id'];
        
        $response = onestream_ApiCall($params, 'POST', "/ext/line/{$lineId}/terminate");
        
        if (isset($response['line_id'])) {
            return 'success';
        }
        
        return $response['error'] ?? 'Failed to terminate line';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'TerminateAccount', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Admin services tab fields
 */
function onestream_AdminServicesTabFields(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        $fields = array();
        
        if ($serviceType == 'credits') {
            $profile = onestream_ApiCall($params, 'GET', '/ext/profile');
            
            if (isset($profile['credits'])) {
                $fields['Reseller Name'] = $profile['name'] ?? 'N/A';
                $fields['Available Credits'] = $profile['credits'];
                $fields['Service Type'] = 'Reseller Credits';
            }
        } else {
            if (!empty($params['username'])) {
                $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
                    'username' => $params['username'],
                    'password' => $params['password'],
                ));
                
                if (isset($findResponse['line_id'])) {
                    $lines = onestream_ApiCall($params, 'GET', '/ext/lines', array(
                        'username' => $params['username'],
                    ));
                    
                    if (is_array($lines) && count($lines) > 0) {
                        $line = $lines[0];
                        
                        $fields['Line ID'] = $line['line_id'];
                        $fields['Username'] = $line['username'];
                        $fields['Type'] = ucfirst($line['type']);
                        $fields['Status'] = $line['is_enabled'] ? 'Enabled' : 'Disabled';
                        $fields['Expire At'] = $line['expire_at'] ?? 'Unlimited';
                        $fields['Max Connections'] = $line['max_connections'];
                        $fields['Package ID'] = $line['package_id'] ?? 'N/A';
                        $fields['Is Trial'] = $line['is_trial'] ? 'Yes' : 'No';
                        
                        if (!empty($line['mac_addr'])) {
                            $fields['MAC Address'] = $line['mac_addr'];
                        }
                    }
                }
            }
        }
        
        return $fields;
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'AdminServicesTabFields', $params, $e->getMessage(), $e->getTraceAsString());
        return array('Error' => $e->getMessage());
    }
}

/**
 * Client area output
 */
function onestream_ClientArea(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        $serviceData = array();
        
        if ($serviceType == 'credits') {
            $profile = onestream_ApiCall($params, 'GET', '/ext/profile');
            if (isset($profile['credits'])) {
                $serviceData = $profile;
            }
        } else {
            if (!empty($params['username'])) {
                $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
                    'username' => $params['username'],
                    'password' => $params['password'],
                ));
                
                if (isset($findResponse['line_id'])) {
                    $lines = onestream_ApiCall($params, 'GET', '/ext/lines', array(
                        'username' => $params['username'],
                    ));
                    
                    if (is_array($lines) && count($lines) > 0) {
                        $serviceData = $lines[0];
                    }
                }
            }
        }
        
        return array(
            'tabOverviewReplacementTemplate' => 'templates/onestream_clientarea.tpl',
            'templateVariables' => array(
                'serviceData' => $serviceData,
                'serviceType' => $serviceType,
                'username' => $params['username'],
                'password' => $params['password'],
                'serverUrl' => rtrim($params['serverhostname'], '/'),
            ),
        );
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'ClientArea', $params, $e->getMessage(), $e->getTraceAsString());
        return array(
            'tabOverviewReplacementTemplate' => 'templates/onestream_error.tpl',
            'templateVariables' => array('error' => $e->getMessage()),
        );
    }
}

/**
 * Admin custom buttons
 */
function onestream_AdminCustomButtonArray()
{
    return array(
        "Get Packages" => "GetPackages",
        "Sync Credentials" => "SyncCredentials",
        "Renew Line" => "RenewLine",
    );
}

/**
 * Sync credentials from 1Stream
 */
function onestream_SyncCredentials(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            return 'Not applicable for credit accounts';
        }
        
        if (empty($params['username'])) {
            return 'No username found';
        }
        
        $lines = onestream_ApiCall($params, 'GET', '/ext/lines', array(
            'username' => $params['username'],
        ));
        
        if (!is_array($lines) || count($lines) == 0) {
            return 'Line not found';
        }
        
        $line = $lines[0];
        
        Capsule::table('tblhosting')
            ->where('id', $params['serviceid'])
            ->update([
                'username' => $line['username'],
                'password' => encrypt($line['password']),
            ]);
        
        return 'success - Synced: ' . $line['username'];
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'SyncCredentials', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Renew a line
 */
function onestream_RenewLine(array $params)
{
    try {
        $serviceType = $params['configoption1'];
        
        if ($serviceType == 'credits') {
            return 'Not applicable for credit accounts';
        }
        
        // Get package ID from dropdown
        $packageId = $params['configoption2'];
        
        if (empty($packageId) || $packageId == 'manual') {
            return 'Package not selected';
        }
        
        if (empty($params['username'])) {
            return 'No username found';
        }
        
        $findResponse = onestream_ApiCall($params, 'GET', '/ext/line/find', array(
            'username' => $params['username'],
            'password' => $params['password'],
        ));
        
        if (!isset($findResponse['line_id'])) {
            return 'Line not found';
        }
        
        $lineId = $findResponse['line_id'];
        
        $postData = array(
            'package' => intval($packageId),
            'rid' => 'whmcs_renew_' . $params['serviceid'] . '_' . time(),
        );
        
        $response = onestream_ApiCall($params, 'POST', "/ext/line/{$lineId}/renew", $postData);
        
        if (isset($response['line_id'])) {
            return 'success - Renewed until: ' . ($response['expire_at'] ?? 'unknown');
        }
        
        return $response['error'] ?? 'Failed to renew';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'RenewLine', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Get available packages
 */
function onestream_GetPackages(array $params)
{
    try {
        $packages = onestream_ApiCall($params, 'GET', '/ext/packages');
        
        if (is_array($packages)) {
            $result = '<div style="font-family: monospace;"><strong>Available Packages:</strong><br><br>';
            foreach ($packages as $pkg) {
                $days = round($pkg['duration_hours'] / 24, 1);
                
                $result .= sprintf(
                    '<div style="margin-bottom: 10px; padding: 8px; background: #f5f5f5; border-left: 3px solid #007bff;">' .
                    '<strong>ID: %d</strong> - %s<br>' .
                    'Cost: %s credits | Duration: %s days | Connections: %d' .
                    '</div>',
                    $pkg['id'],
                    htmlspecialchars($pkg['name']),
                    $pkg['price_credits'],
                    $days,
                    $pkg['max_connections']
                );
            }
            $result .= '</div>';
            return $result;
        }
        
        return 'Failed to retrieve packages';
        
    } catch (Exception $e) {
        logModuleCall('onestream', 'GetPackages', $params, $e->getMessage(), $e->getTraceAsString());
        return $e->getMessage();
    }
}

/**
 * Make API call to 1Stream B2B API
 */
function onestream_ApiCall($params, $method, $endpoint, $data = array())
{
    $baseUrl = $params['serverhostname'];
    
    // Add http:// if missing
    if (!preg_match('/^https?:\/\//', $baseUrl)) {
        $baseUrl = 'http://' . $baseUrl;
    }
    
    $baseUrl = rtrim($baseUrl, '/');
    $apiKey = $params['serveraccesshash'];
    $authUser = $params['serverpassword'];
    
    $url = $baseUrl . $endpoint;
    
    if ($method == 'GET' && !empty($data)) {
        $url .= '?' . http_build_query($data);
    }
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 1);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $headers = array(
        'X-Api-Key: ' . $apiKey,
        'X-Auth-User: ' . $authUser,
        'Content-Type: application/json',
    );
    
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    if ($method == 'POST') {
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('cURL Error: ' . $error);
    }
    
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        $result = array(
            'error' => 'Invalid JSON response',
            'raw_response' => $response,
            'http_code' => $httpCode,
        );
    }
    
    logModuleCall('onestream', $method . ' ' . $endpoint, array(
        'url' => $url,
        'data' => $data,
    ), $response, $result);
    
    return $result;
}